/*
 *  Copyright(c) 2003. Your Techonology Partner(YTP). All rights reserved.
 *  
 *  このプログラムの著作権はYour Techonology Partner(YTP)が保有します。
 *  このプログラムはアパッチソフトウェアライセンスに従って配布します。
 *  このプログラムを再配布あるいは改造する場合は、上記著作権表示を必ず
 *  含めるようにして下さい。免責事項も同ライセンスに準じます。詳細は
 *  http://www.apache.org/LICENSE を参照して下さい。
 *  
 *  Your Techonology Partner(YTP) owns the copyright of this program.
 *  This program is provided in conformity with The Apache Software
 *  License agreement. Redistribution and reproduction must contain
 *  the above copyright notice. The Disclaimer is also based on
 *  The Apache Software License. Redistributor can refer to
 *  http://www.apache.org/LICENSE for further details.
 */

package jp.ne.ytp.util;

import java.util.*;

/**
 *  文字列を、カンマや指定された区切り文字で1項目ずつのStringに分解します。<br>
 *  java.util.StringTokenizerクラスとの大きな違いは、区切り文字のみが続いても、
 *  切り出す文字列を空文字("")として返すことです。<br>
 *  また、ダブルクォーテーションで囲まれた文字列は、その中に区切り文字を含んでいても、
 *  一つのトークンとみなします。<br>
 *  【例】: 『abc,,"efg,hi",jk』は、「abc」「空文字」「efg,hi」「jk」の4つのトークンとして返します。
 *  @version $Id: CsvTokenizer.java,v 1.1.1.1 2003/02/14 15:56:10 YT0050 Exp $
 *  @author YTP
 */
public class CsvTokenizer {
    /**
     *  カンマです。
     */
    public static final char COMMA = ',';
    
    /**
     *  ダブルクォーテーションです。
     */
    private static final char QUOTA = '"';
    
    /**
     *  ホワイトスペースです。
     */
    private static final char SPACE = ' ';
    
    /**
     *  分解対象の文字列です。
     */
    private String sTokens_;
    
    /**
     *  指定された区切り文字です。
     */
    private char cDelimit_;
    
    /**
     *  切り出したトークンを保持します。
     */
    private List lstTokens_ = null;
    
    /**
     *  lstTokens_が保持するトークンの数です。
     */
    private int iTokenNum_ = 0;
    
    /**
     *  現在のトークンの位置です。
     */
    private int iIndex_ = 0;
    
    /**
     *  空文字列("")を持つインスタンスを生成します。区切り文字はカンマとなります。
     */
    public CsvTokenizer() {
        this("");
    }
    
    /**
     *  カンマを区切り文字とする、sTokens分解のためのインスタンスを生成します。
     *  @param sTokens 分解対象の文字列
     */
    public CsvTokenizer(String sTokens) {
        setTokens(sTokens);
    }
    
    /**
     *  cDelimitで指定された文字を区切り文字とする、
     *  sTokens分解のためのインスタンスを生成します。
     *  @param sTokens 分解対象の文字列
     *  @param cDelimit 区切り文字
     */
    public CsvTokenizer(String sTokens, char cDelimit) {
        setTokens(sTokens, cDelimit);
    }
    
    /**
     *  分解する対象としてsTokensを設定します。区切り文字はカンマとなります。
     *  @param sTokens 分解対象の文字列
     */
    public void setTokens(String sTokens) {
        setTokens(sTokens, COMMA);
    }
    
    /**
     *  分解する対象としてsTokensを、区切り文字としてcDelimitをそれぞれ設定します。
     *  @param sTokens 分解対象の文字列
     *  @param cDelimit 区切り文字
     */
    public void setTokens(String sTokens, char cDelimit) {
        sTokens_ = sTokens;
        cDelimit_ = cDelimit;
        lstTokens_ = new ArrayList();
        iIndex_ = 0;
        iTokenNum_ = 0;
        breakTokens(sTokens);
    }
    
    /**
     *  分解対象文字列を返します。
     *  @return 分解対象文字列
     */
    public String getTokens() {
        return sTokens_;
    }
    
    /**
     *  区切り文字を返します。
     *  @return 区切り文字
     */
    public char getDelimiter() {
        return cDelimit_;
    }
    
    /**
     *  文字列を区切り文字で1項目ずつStringに分解し、lstTokens_に格納します。
     *  @param  sCsvString 分解対象の文字列
     */
    private void breakTokens(String sCsvString) {
        int iStatus = 0;        // オートマトンの状態
        int iBegPos = 0;        // トークンの開始位置
        int iEndPos = 0;        // トークンの終了位置
        char[] cChars = null;
        
        cChars = new char[sCsvString.length()];
        sCsvString.getChars(0, sCsvString.length(), cChars, 0);
        
        // 文字数分処理する
        for (int iI = 0; iI < cChars.length; iI++) {
            // オートマトンの状態を判定
            switch (iStatus) {
                case 0:
                    if (cChars[iI] == cDelimit_) {
                        // トークンの文字が無い
                        lstTokens_.add("");
                        iEndPos++;
                        iBegPos = iEndPos;  // 開始位置のリセット
                        iStatus = 2;
                    } else if (cChars[iI] == QUOTA) {
                        iEndPos++;
                        iBegPos = iEndPos;  // 開始位置のリセット
                        iStatus = 3;
                    } else {
                        iEndPos++;
                        iStatus = 1;
                    }
                    break;
                case 1:
                    if (cChars[iI] == cDelimit_) {
                        // 文字列をリストに追加する
                        lstTokens_.add(new String(cChars, iBegPos, iEndPos - iBegPos));
                        iEndPos++;
                        iBegPos = iEndPos;  // 開始位置のリセット
                        iStatus = 2;
                    } else {
                        iEndPos++;
                        // 状態はそのまま
                    }
                    
                    break;
                case 2:
                    if (cChars[iI] == cDelimit_) {
                        // トークンの文字が無い
                        lstTokens_.add("");
                        iEndPos++;
                        iBegPos = iEndPos;  // 開始位置のリセット
                        iStatus = 2;
                    } else if (cChars[iI] == QUOTA) {
                        iEndPos++;
                        iBegPos = iEndPos;  // 開始位置のリセット
                        iStatus = 3;
                    } else {
                        iEndPos++;
                        iStatus = 1;
                    }
                    break;
                case 3:
                    if (cChars[iI] == QUOTA) {
                        // 文字列をリストに追加する
                        lstTokens_.add(new String(cChars, iBegPos, iEndPos - iBegPos));
                        iEndPos++;
                        iStatus = 4;
                    } else {
                        iEndPos++;
                        // 状態はそのまま
                    }
                    break;
                case 4:
                    if (cChars[iI] == cDelimit_) {
                        iEndPos++;
                        iBegPos = iEndPos;  // 開始位置のリセット
                        iStatus = 2;
                    } else if (cChars[iI] == SPACE) {
                        iEndPos++;
                    } else {
                        // この状態はエラー
                        iEndPos++;
                    }
                    break;
            }
        }
        
        // オートマトンの終了状態を判定
        switch (iStatus) {
            case 0:
                break;
            case 1:
                // 文字列をリストに追加する
                lstTokens_.add(new String(cChars, iBegPos, iEndPos - iBegPos));
                break;
            case 2:
                // トークンの文字が無い
                lstTokens_.add("");
                break;
            case 3:
                break;
            case 4:
                break;
            default:
                break;
        }
        // リストに入っているトークンの数を取得
        iTokenNum_ = lstTokens_.size();
    }
    
    /**
     *  トークナイザの文字列で利用できるトークンがまだあるかどうか判定します。
     *  このメソッドがtrueを返す場合、{@link CsvTokenizer#nextToken()}の呼出は適切にトークンを返します。
     *  @return boolean true:次のトークンが存在する false:これ以上トークンが存在しない
     */
    public boolean hasMoreTokens() {
        if (iIndex_ < iTokenNum_ ) {
            return true;
        } else {
            return false;
        }
    }
    
    /**
     *  次のトークンを返します。
     *  @return String トークン文字列
     *  @exception NoSuchElementException トークンが残っていない場合
     */
    public String nextToken() throws NoSuchElementException {
        try {
            return (String)lstTokens_.get(iIndex_++);
        } catch (ArrayIndexOutOfBoundsException e) {
            throw new NoSuchElementException();
        }
    }
    
    /**
     *  トークン取得用のインデックスを0に戻します。<br>
     *  {@link CsvTokenizer#hasMoreTokens()}メソッドでfalseが返された後で当メソッドを呼び出すと、
     *  分解された最初のトークンから再び取得可能になります。
     */
    public void reset() {
        iIndex_ = 0;
    }
}
