/*
 *  Copyright(c) 2003. Your Techonology Partner(YTP). All rights reserved.
 *  
 *  このプログラムの著作権はYour Techonology Partner(YTP)が保有します。
 *  このプログラムはアパッチソフトウェアライセンスに従って配布します。
 *  このプログラムを再配布あるいは改造する場合は、上記著作権表示を必ず
 *  含めるようにして下さい。免責事項も同ライセンスに準じます。詳細は
 *  http://www.apache.org/LICENSE を参照して下さい。
 *  
 *  Your Techonology Partner(YTP) owns the copyright of this program.
 *  This program is provided in conformity with The Apache Software
 *  License agreement. Redistribution and reproduction must contain
 *  the above copyright notice. The Disclaimer is also based on
 *  The Apache Software License. Redistributor can refer to
 *  http://www.apache.org/LICENSE for further details.
 */

package jp.ne.ytp.util;

import java.util.*;

/**
 *  複数階層のキーを持つことが可能なハッシュテーブルです。
 *  インスタンスの値として{@link java.util.HashMap}
 *  インスタンスを入れ子で持つことで複数階層を実装しています。<br/>
 *  キーの階層数はコンストラクタによって決定します。一度決めたキーの数は変更できません。<br/>
 *  次のようなコードを書いた場合、
 *  <pre>
 *      String[] s1 = {"aa", "bbb", "111"};
 *      String[] s2 = {"aa", "ccc", "222"};
 *      String[] s3 = {"aa", "bbb", "222"};
 *      MultiHash hash = new MultiHash(3);
 *      
 *      hash.put(s1, "1番");
 *      hash.put(s2, "2番");
 *      hash.put(s3, "3番");
 *      
 *      System.out.println(hash.get(s2));
 *  </pre>
 *  結果は「2番」と表示されます。<br>
 *  またさらに以下を追加した場合は、
 *  <pre>
 *      String[] key = {"aa", "bbb"};
 *      System.out.println(hash.keySet(key));
 *      System.out.println(hash.values(key));
 *  </pre>
 *  [222, 111]<br>
 *  [3番, 1番]<br>
 *  と表示されます。
 *  @see java.util.HashMap
 *  @version $Id: MultiKeyHash.java,v 1.1 2003/02/19 13:39:36 YT0050 Exp $
 *  @author YTP
 */
public class MultiKeyHash {
    /**
     *  キーの階層数です。
     */
    private final int iLayer_;
    
    /**
     *  最上位階層(ルート)のMapです。
     */
    private HashMap mRoot_ = new HashMap();
    
    /**
     *  キーが1階層だけのインスタンスを生成します。
     *  インスタンス生成後この階層数は変更できません。
     */
     public MultiKeyHash() {
        this(1);
     }
    
    /**
     *  iLayerで指定された階層数をキーとして持つインスタンスを生成します。
     *  インスタンス生成後この階層数は変更できません。
     */
     public MultiKeyHash(int iLayer) {
        iLayer_ = iLayer;
     }
    
    /**
     *  keysをキーとして、valueを格納します。<br>
     *  keysで指定された配列数がコンストラクタで指定した階層数以外の場合、
     *  IllegalArgumentExceptionを投げます。
     *  @param keys 格納対象データのキーを格納した配列
     *  @param value 格納対象データ
     *  @exception IllegalArgumentException 配列の要素数が階層数以外の場合。
     */
    public void put(Object[] keys, Object value) {
        // 配列数が階層数でない場合
        if (keys.length != iLayer_) {
            throw new IllegalArgumentException();
        }
        
        HashMap mCurrent = mRoot_;
        int i = 0;
        
        // キーの数-1 だけ M010604 YT0050
        while (i < keys.length - 1) {
            Object lower = mCurrent.get(keys[i]);
            
            // 新規キーの場合
            if (lower == null) {
                HashMap mNew = new HashMap();
                
                // 次のレベルのハッシュを定義する
                mCurrent.put(keys[i], mNew);
                // カレントのハッシュを次のレベルに下げる
                mCurrent = mNew;
            } else {
                mCurrent = (HashMap)lower;
            }
            i++;
        }
        // データの格納
        mCurrent.put(keys[i], value);
    }
    
    /**
     *  データを検索します。見つからない場合はnullを返します。<br>
     *  keys[keys.length - 1]配下のデータを検索しようとしますが、
     *  最下層にたどり着く途中の階層でキーが見つからなかった場合は、
     *  その時点で検索をやめnullを返します。
     *  また、キーの階層数を超えた配列数がkeysで指定された場合もnullを返します。
     *  @param keys 検索対象のキーを格納した配列
     *  @return 検索したデータ
     */
    public Object get(Object[] keys) {
        // 階層数が最下層を超えた場合
        if (keys.length > iLayer_) {
            return null;
        }
        
        HashMap mCurrent = mRoot_;
        int i = 0;
        
        // キーの数-1 だけ M010604 YT0050
        while (i < keys.length - 1) {
            Object lower = mCurrent.get(keys[i]);
            
            // キーが見つからない場合
            if (lower == null) {
                return null;
            } else {
                mCurrent = (HashMap)lower;
                i++;
            }
        }
        
        return mCurrent.get(keys[i]);
    }
    
    /**
     *  keysで指定されたキー配下(1階層下)のMapが保持するキーの一覧を返します。<br>
     *  返すセットは当クラス内のマップと連動するので、
     *  セットに対する変更はマップに反映されます。
     *  セットに対する反復の処理中にマップが変更された場合、反復の結果は保証されません。<br>
     *  指定のキーが見つからなかった場合はnullを返します。
     *  また、キーの階層数-1を超えた配列数がkeysで指定された場合もnullを返します。
     *  @param keys 検索対象のキーを格納した配列
     *  @return キーの一覧
     *  @see java.util.Map#keySet()
     */
    public Set keySet(Object[] keys) {
        // 階層数が最下層-1を超えた場合
        if (keys.length > iLayer_ - 1) {
            return null;
        }
        
        HashMap mTarget = (HashMap)this.get(keys);
        
        // 対象が見つかった場合
        if (mTarget != null) {
            return mTarget.keySet();
        } else {
            return null;
        }
    }
    
    /**
     *  keysで指定されたキー配下(1階層下)のMapが保持する値の一覧を返します。<br>
     *  返すコレクションは当クラス内のマップと連動するので、
     *  コレクションに対する変更はマップに反映されます。
     *  コレクションに対する反復の処理中にマップが変更された場合、反復の結果は保証されません。<br>
     *  指定のキーが見つからなかった場合はnullを返します。
     *  また、キーの階層数-1を超えた配列数がkeysで指定された場合もnullを返します。
     *  @param keys 検索対象のキーを格納した配列
     *  @return 値の一覧
     *  @see java.util.Map#values()
     */
    public Collection values(Object[] keys) {
        // 階層数が最下層-1を超えた場合
        if (keys.length > iLayer_ - 1) {
            return null;
        }
        
        HashMap mTarget = (HashMap)this.get(keys);
        
        // 対象が見つかった場合
        if (mTarget != null) {
            return mTarget.values();
        } else {
            return null;
        }
    }
}
