/*
 *  Copyright(c) 2016. Your Techonology Partner(YTP). All rights reserved.
 *  
 *  このプログラムの著作権はYour Techonology Partner(YTP)が保有します。
 *  このプログラムはアパッチソフトウェアライセンスに従って配布します。
 *  このプログラムを再配布あるいは改造する場合は、上記著作権表示を必ず
 *  含めるようにして下さい。免責事項も同ライセンスに準じます。詳細は
 *  http://www.apache.org/LICENSE を参照して下さい。
 *  
 *  Your Techonology Partner(YTP) owns the copyright of this program.
 *  This program is provided in conformity with The Apache Software
 *  License agreement. Redistribution and reproduction must contain
 *  the above copyright notice. The Disclaimer is also based on
 *  The Apache Software License. Redistributor can refer to
 *  http://www.apache.org/LICENSE for further details.
 */

package jp.ne.ytp.util;

import java.util.List;
import java.util.Map;
import java.util.Stack;

/**
 *  いわゆる逆ポーランド記法(後置記法)を実現します。
 *  渡された計算式を、指定の演算子とその優先順位に従って逆ポーランド記法に変換します。
 *  演算子や優先順位を利用する側で全て決定できるため、
 *  当クラスの利用は四則演算に限定されません。
 *  「カッコ()」の優先順位も外部で指定する必要がありますが、
 *  一般的な利用をする場合、<a href="#OPENPBRACKET">開きカッコ</a>は最高位、
 *  <a href="#CLOSEBRACKET">閉じカッコ</a>は最低位として設定してください。<br/>
 *  次のコードは四則演算を実現する簡単な例です。<br/>
 *  <pre>
 *      ArrayList before = new ArrayList();
 *      before.add("1");
 *      before.add("+");
 *      before.add("2");
 *      before.add("*");
 *      before.add("a");
 *      
 *      HashMap operators = new HashMap();
 *      operators.put(ReversePolish.OPENPBRACKET, new Integer(50));
 *      operators.put(ReversePolish.OPERAND, new Integer(40));
 *      operators.put("*", new Integer(30));
 *      operators.put("/", new Integer(30));
 *      operators.put("+", new Integer(20));
 *      operators.put("-", new Integer(20));
 *      operators.put(ReversePolish.CLOSEBRACKET, new Integer(0));
 *      
 *      ReversePolish rp = new ReversePolish(operators);
 *      List reversed = rp.parse2Polish(before);
 *      
 *      System.out.print("Reversed:");
 *      for (int i = 0; i < reversed.size(); i++) {
 *          System.out.print(reversed.get(i));
 *      }
 *      System.out.println("");
 *  </pre>
 *  この結果は、<code>Reversed:12a*+</code> と表示されます。
 *  @version $Id: ReversePolish.java,v 1.2 2003/02/19 18:38:04 YT0050 Exp $
 *  @author YTP
 */
public class ReversePolish {
    /**
     *  オペランドを意味します。優先順位を指定する際に使います。
     */
    public static final String OPERAND = "$OPERAND";
    
    /**
     *  開きカッコ"("です。優先順位を指定する際に使います。
     */
    public static final String OPENPBRACKET = "(";
    
    /**
     *  閉じカッコ")"です。優先順位を指定する際に使います。
     */
    public static final String CLOSEBRACKET = ")";
    
    /**
     *  演算子と優先順位を格納します。
     */
    private Map<String, Integer> mOperatorPriority_ = null;
    
    /**
     *  デフォルトコンストラクタです。
     *  このコンストラクタで生成されたインスタンスでは、
     *  {@link ReversePolish#setOperator(Map)}メソッドを必ず呼び出して下さい。
     *  @see ReversePolish#setOperator(Map)
     */
     public ReversePolish() {
        super();
     }
     
    /**
     *  operatorPriorityで指定された演算子と優先順位を持つインスタンスを生成します。
     *  operatorPriorityのキーには演算子を表すStringを、
     *  値には優先順位を表すIntegerクラスインスを、それぞれ設定してください。
     *  @param operatorPriority 演算子とその優先順位を設定したMap
     *  @exception IllegalArgumentException operatorPriorityがnullの場合
     */
    public ReversePolish(Map<String, Integer> operatorPriority) throws IllegalArgumentException {
        super();
        this.setOperator(operatorPriority);
    }
     
    /**
     *  演算子と優先順位を設定します。
     *  operatorPriorityには、キーに演算子を表すStringを、
     *  値に優先順位を表すIntegerクラスインスを、それぞれ設定してください。
     *  @param operatorPriority 演算子とその優先順位を設定したMap
     *  @exception IllegalArgumentException operatorPriorityがnullの場合
     */
    public void setOperator(Map<String, Integer> operatorPriority) throws IllegalArgumentException {
        if (operatorPriority != null) {
            mOperatorPriority_ = operatorPriority;
        } else {
            throw new IllegalArgumentException("Operand is null.");
        }
    }
    
    /**
     *  tokensに設定された演算式を逆ポーランド記法に変換します。
     *  tokensの各要素はtoString()メソッドにて、
     *  オペランドまたは演算子を返すようにしておく必要があります。
     *  またequals()メソッドを実装し、
     *  オブジェクト同士の比較が正しくできるようにもしておいてください。
     *  (要素としてStringクラスを使うのが一般的です)。
     *  逆ポーランド記法に変換したオペランドまたは演算子を、
     *  Stringクラスの要素としてリストに一つずつ格納し返します。
     *  @param tokens オペランドまたは演算子を要素に持つリスト
     *  @return 逆ポーランド記法に変換したトークン(Stringクラス)を持つリスト
     */
    public Stack<String> parse2Polish(List<String> tokens) {
        // 逆ポーランド記法で格納
        List<String> reversed = new Stack<String>();
        // 一時スタック
        Stack<String> stackTokens = new Stack<String>();
        String sToken = null;
        int iTokenPriority = -1;
        
        // トークンの数だけ
        for (int i = 0; i < tokens.size(); i++) {
            //System.out.println("tokens[" + i + "]=" + tokens.get(i).toString());
            // スタックが空でない
            if (!stackTokens.empty()) {
                // 現在のトークンの優先順位取得
                iTokenPriority = getPriority(tokens.get(i).toString());
                // スタックトップトークンの優先順位取得
                //iStackPriority = getPriority((String)stackTokens.peek());
                
                /* スタックが空でない
                 * かつ 今のトークンの優先順位 <= スタックトップトークンの優先順位
                 * かつ スタックトップのトークンが左括弧でない
                 */
                while (!stackTokens.empty()
                        && iTokenPriority <= getPriority((String)stackTokens.peek())
                        && !stackTokens.peek().equals(OPENPBRACKET)) {
                    sToken = (String)stackTokens.pop();
                    // 現在のトークンが左括弧でない場合
                    if (!sToken.equals(OPENPBRACKET)) {
                        // 逆ポーランド記法のベクターに積む
                        reversed.add(sToken);
                        //System.out.println("Polish token=" + sToken);
                    }
                }
            }
            // トークンが右括弧ならば
            if (tokens.get(i).equals(CLOSEBRACKET)) {
                // 左括弧を捨てる
                sToken = (String)stackTokens.pop();
                // System.out.println("Trashed token=" + sToken);
            } else {
                // スタックに積む
                stackTokens.push(tokens.get(i));
                // System.out.println("Stacked token=" + tokens.elementAt(i));
            }
        }
        // スタックに残っているトークンを逆ポーランド記法のベクターに全て入れる
        while (!stackTokens.empty()) {
            sToken = (String)stackTokens.pop();
            if (!sToken.equals(OPENPBRACKET)) {
                reversed.add(sToken);
                // System.out.println("Polish token=" + sToken);
            }
        }
        
        return (Stack<String>)reversed;
    }
    
    /**
     *  トークンの優先順位を取得します。
     *  @param sToken トークン文字列
     *  @return トークンの優先順位
     */
    private int getPriority(String sToken) {
        // トークンがオペランドの場合
        if (mOperatorPriority_.get(sToken) == null) {
            //System.out.println("Priority[" + sToken + "]=" + ((Integer)(mOperatorPriority_.get(OPERAND))).intValue());
            return ((Integer)(mOperatorPriority_.get(OPERAND))).intValue();
        } else {
            //System.out.println("Priority[" + sToken + "]=" + ((Integer)(mOperatorPriority_.get(sToken))).intValue());
            return ((Integer)(mOperatorPriority_.get(sToken))).intValue();
        }
    }
}
