/*
 *  Copyright(c) 2016. Your Technology Partner(YTP). All rights reserved.
 *  
 *  このプログラムの著作権はYour Technology Partner(YTP)が保有します。
 *  このプログラムはアパッチソフトウェアライセンスに従って配布します。
 *  このプログラムを再配布あるいは改造する場合は、上記著作権表示を必ず
 *  含めるようにして下さい。免責事項も同ライセンスに準じます。詳細は
 *  http://www.apache.org/LICENSE を参照して下さい。
 *  
 *  Your Technology Partner(YTP) owns the copyright of this program.
 *  This program is provided in conformity with The Apache Software
 *  License agreement. Redistribution and reproduction must contain
 *  the above copyright notice. The Disclaimer is also based on
 *  The Apache Software License. Redistributor can refer to
 *  http://www.apache.org/LICENSE for further details.
 */

package jp.ne.ytp.util.tree;

import java.util.*;
import jp.ne.ytp.util.*;

/**
 *  多階層型二分探索木のノードです。
 *  各ノードインスタンスは次の情報を持ちます。
 *  <ol>
 *      <li>左右ノードへの枝</li>
 *      <li>下階層への枝</li>
 *      <li>階層番号を表す1オリジンの整数</li>
 *      <li>ノードのキーを表現する文字列</li>
 *      <li>ノードのデータを表現するオブジェクト</li>
 *      <li>ノードキーの大小比較をするためのコンパレータ</li>
 *  </ol>
 *  @see TreeNode
 *  @author YTP
 */
class MultiKeyTreeNode extends TreeNode {
    /**
     *  下階層のノードです。
     */
    MultiKeyTreeNode lowerNode;
    
    /**
     *  ノードの階層番号です。
     */
    private int iLayer_;
    
    /**
     *  ノードのデータオブジェクトです。
     */
    private Object value_;
    
    /**
     *  ノードキー大小比較用のコンパレータです。
     */
    private Comparator<Object> comp_ = null;
    
    /**
     *  ノードのキーとしてkeyを、階層としてiLayerを持つノードインスタンスを生成します。
     *  コンパレータは標準コンパレータ{@link DefaultComparator}を使用します。
     *  @param key キー
     *  @param iLayer 階層番号
     */
    MultiKeyTreeNode(Object key, int iLayer) {
        this(key, iLayer, DefaultComparator.getComparator());
    }
    
    /**
     *  ノードのキーとしてkeyを、階層としてiLayerを、
     *  コンパレータとしてcompを持つノードインスタンスを生成します。
     *  @param key キー
     *  @param iLayer 階層番号
     *  @param comp コンパレータ
     */
    MultiKeyTreeNode(Object key, int iLayer, Comparator<Object> comp) {
        key_ = key;
        iLayer_ = iLayer;
        comp_ = comp;
    }
    
    /**
     *  ノードキー大小比較用のコンパレータを設定します。
     *  @param comp コンパレータ
     */
    void setComparator(Comparator<Object> comp) {
        comp_ = comp;
    }
    
    /**
     *  このインスタンスのキーとcompNodeのキーを比較した結果を返します。
     *  比較には標準コンパレータ({@link DefaultComparator})あるいは、
     *  コンストラクタ{@link MultiKeyTreeNode(Object, int, Comparator)}または
     *  {@link MultiKeyTreeNode#setComparator(Comparator)}で設定されたコンパレータを使用します。
     *  @param compNode 比較対照のノード
     */
    int compare(TreeNode compNode) {
        return comp_.compare(this.key_, compNode.key_); // fixed forgotten key. 20160505
    }
    
    /**
     *  このノードインスタンスが管理するデータオブジェクトを設定します。
     *  @param value データオブジェクト
     */
    void setData(Object value) {
        value_ = value;
    }
    
    /**
     *  このノードインスタンスが管理するデータオブジェクトを返します。
     */
    Object getData() {
        return value_;
    }
    
    /**
     *  このノードインスタンスのキーを返します。
     */
    Object getKey() {
        return key_;
    }
    
    /**
     *  このノードインスタンスのキーの階層番号を1オリジンで返します。
     *  @return このノードの階層番号
     */
    int getLayer() {
        return iLayer_;
    }
    
    /**
     *  このノードキーの文字列表現を返します。
     *  @return このノードキーの文字列表現
     */
    public String toString() {
        return key_.toString();
    }
}
