/*
 *  Copyright(c) 2016. Your Technology Partner(YTP). All rights reserved.
 *  
 *  このプログラムの著作権はYour Technology Partner(YTP)が保有します。
 *  このプログラムはアパッチソフトウェアライセンスに従って配布します。
 *  このプログラムを再配布あるいは改造する場合は、上記著作権表示を必ず
 *  含めるようにして下さい。免責事項も同ライセンスに準じます。詳細は
 *  http://www.apache.org/LICENSE を参照して下さい。
 *  
 *  Your Technology Partner(YTP) owns the copyright of this program.
 *  This program is provided in conformity with The Apache Software
 *  License agreement. Redistribution and reproduction must contain
 *  the above copyright notice. The Disclaimer is also based on
 *  The Apache Software License. Redistributor can refer to
 *  http://www.apache.org/LICENSE for further details.
 */

package jp.ne.ytp.util.tree;

/**
 *  {@link MultiKeyTree}クラスからイベントを通知するためのメソッドを規定するインタフェースです。
 *  {@link MultiKeyTree}クラスを利用する場合は、このインタフェースを実装したクラスが必要です。<br>
 *  大きく分けると、
 *  <ol>
 *  <li>木を作成する際に呼び出されるメソッド
 *      <ul>
 *          <li>{@link TreeActionListener#initializeRoot()}</li>
 *          <li>{@link TreeActionListener#create(int)}</li>
 *          <li>{@link TreeActionListener#match(int, Object)}</li>
 *      </ul>
 *   </li>
 *  <li>木を走査(トラバース)する際に呼び出されるメソッド
 *      <ul>
 *          <li>{@link TreeActionListener#makeHeader(Object, int, Object)}</li>
 *          <li>{@link TreeActionListener#sort(Object[], int, Object)}</li>
 *          <li>{@link TreeActionListener#makeFooter(Object, int, Object)}</li>
 *      </ul>
 *   </li>
 *  </ol>
 *  の2種類があります。
 *  @see MultiKeyTree
 *  @author YTP
 */
public interface TreeActionListener extends java.util.EventListener {
    /*** 木作成時の処理 ***/
    /**
     *  ルートノードを初期化するために呼び出されます。
     *  このメソッドを実装する場合、ルートノード(階層番号0)用のデータオブジェクトを生成し、
     *  そのオブジェクトを返すようにして下さい。
     *  このメソッドは、{@link MultiKeyTree}クラスを利用する側が、
     *  {@link MultiKeyTree#addListener(TreeActionListener)}を呼んだ時に一度だけ呼び出されます。
     *  @return ルートノードが持つデータオブジェクト
     */
    public Object initializeRoot();
    
    /**
     *  まだ存在しないキーを持つノードが新たに生成された場合に呼び出されるメソッドです。
     *  このメソッドを実装する場合、iLayerを判断して適切なデータオブジェクトを生成し、
     *  そのオブジェクトを返すようにして下さい。
     *  各階層で新規のノード(キー)が生成された場合は、
     *  このメソッドが必ず呼び出されることに注意して下さい。
     *  @param iLayer 生成したノードの階層番号(1オリジン ただしルートノードの場合のみ0)
     *  @return ノードが管理するデータオブジェクト
     */
    public Object create(int iLayer);
    
    /**
     *  同一階層に同一キーのノードがすでに存在する場合に呼び出されます。
     *  このメソッドを実装する場合、iLayerを判断して、
     *  nodeDataで渡されたデータオブジェクトに対して適切な処理を行い、
     *  再びそのオブジェクトを返すようにして下さい。
     *  階層番号が0であるルートノードを含み、
     *  各階層でこのメソッドが呼び出されることに注意して下さい。
     *  @param iLayer 一致したノードの階層番号(1オリジン ただしルートノードの場合のみ0)
     *  @param nodeData ノードが管理するデータオブジェクト
     *  @return ノードが管理するデータオブジェクト
     */
    public Object match(int iLayer, Object nodeData);
    
    /*** トラバース時の処理 ***/
    /**
     *  木の走査時に最下層のノードで呼び出されるメソッドです。
     *  keysは1レコード分に相当する全階層のキー配列です。
     *  nodeDataは最下層ノードが管理するデータオブジェクトです。
     *  {@link TreeActionListener#makeHeader(Object, int, Object)}と
     *  {@link TreeActionListener#makeHeader(Object, int, Object)}を使わずに、
     *  このメソッドのみを実装すると、ソートされた明細として出力することが可能となります。
     *  @param keys 最上位階層から最下層までのキー配列
     *  @param iLayer 最下層の階層数
     *  @param nodeData 最下層ノードが管理するデータオブジェクト
     */
    public void sort(Object[] keys, int iLayer, Object nodeData);
    
    /**
     *  木の走査時に、最下層を除く各ノードのヘッダ部分(行きがけ)で呼び出されるメソッドです。
     *  @param key ノードのキー
     *  @param iLayer ノードの階層番号(1オリジン、ただしルートノードの場合は0)
     *  @param nodeData ノードが管理するデータオブジェクト
     */
    public void makeHeader(Object key, int iLayer, Object nodeData);
    
    /**
     *  木の走査時に、最下層を除く各ノードのフッタ部分(帰りがけ)で呼び出されるメソッドです。
     *  @param key ノードのキー
     *  @param iLayer ノードの階層番号(1オリジン、ただしルートノードの場合は0)
     *  @param nodeData ノードが管理するデータオブジェクト
     */
    public void makeFooter(Object key, int iLayer, Object nodeData);
}
