/*
 *  Copyright(c) 2016. Your Technology Partner(YTP). All rights reserved.
 *  
 *  このプログラムの著作権はYour Technology Partner(YTP)が保有します。
 *  このプログラムはアパッチソフトウェアライセンスに従って配布します。
 *  このプログラムを再配布あるいは改造する場合は、上記著作権表示を必ず
 *  含めるようにして下さい。免責事項も同ライセンスに準じます。詳細は
 *  http://www.apache.org/LICENSE を参照して下さい。
 *  
 *  Your Technology Partner(YTP) owns the copyright of this program.
 *  This program is provided in conformity with The Apache Software
 *  License agreement. Redistribution and reproduction must contain
 *  the above copyright notice. The Disclaimer is also based on
 *  The Apache Software License. Redistributor can refer to
 *  http://www.apache.org/LICENSE for further details.
 */

package jp.ne.ytp.util.tree;

import java.util.*;

/**
 *  二分探索木のノード用基底クラスです。
 *  左右ノードの参照のみを持ちます。<br>
 *  [未解決]単純な二分探索木であるためソートされたデータが来ると、
 *  深刻な性能問題が起きます。赤黒木などの平衡木に換装する必要があります。
 *  @see MultiKeyTreeNode
 *  @author YTP
 */
abstract class TreeNode {
    /**
     * このノードよりキーが小さい場合に返す値です。
     */
    static final int SMALLER = 1;
    
    /**
     * このノードよりキーが大きい場合に返す値です。
     */
    static final int BIGGER = 2;
    
    /**
     * このノードとキーが同じ場合に返す値です。
     */
    static final int BINGO = 0;
    
    /**
     * キーが見つからない場合に返す値です。
     */
    static final int NOTFOUND = -1;
    
    /**
     * このインスタンスノードの左ノードです。
     */
    TreeNode left_; // transfered from MultiKeyNode. 20160505
    
    /**
     * このインスタンスノードの右ノードです。
     */
    TreeNode right_;
    
    /**
     *  ノードキーです。
     */
    Object key_;
    
    /**
     * ノード同士のキーを比較する際に使用する抽象化メソッドです。
     */
    abstract int compare(TreeNode trNode);
    
    /**
     *  newNodeノードを追加します。<br>
     *  newNodeのキーとインスタンスのキーをコンパレータにより比較し、
     *  <ol>
     *      <li>このインスタンスより小さいキーの場合
     *          <ul>
     *              <li>このインスタンスの左ノードが既にある場合はそのノード</li>
     *              <li>このインスタンスの左ノードがまだ無い場合はnewNodeを左にぶら下げてnull</li>
     *          </ul>
     *      </li>
     *      <li>このインスタンスより大きいキーの場合
     *          <ul>
     *              <li>このインスタンスの右ノードが既にある場合はそのノード</li>
     *              <li>このインスタンスの右ノードがまだ無い場合はnewNodeを右にぶら下げてnull</li>
     *          </ul>
     *      </li>
     *      <li>このインスタンスと同じキーの場合はnull</li>
     *  </ol>
     *  をそれぞれ返します。<br>
     *  当メソッドは、内部で再帰を使用していません。
     *  そのため、呼び出し側で結果を判定しながらループさせる必要があります。
     *  @param newNode 追加対象のノード
     *  @return 追加を完了した場合はnull、引き続き処理が必要な場合はそのノード
     */
    TreeNode append(TreeNode newNode) {
        int iComparison;
        
        iComparison = compare(newNode);
        
        // このノードと同じ
        if (iComparison == 0) {
            // 追加終了
            return null;
        // このノードより小さい
        } else if (iComparison > 0) {
            // 左ノードがぶら下がっている
            if (left_ != null) {
                return left_;
            } else {
                left_ = newNode;
                // 追加終了
                return null;
            }
        // このノードより大きい
        } else {
            // 右ノードがぶら下がっている
            if (right_ != null) {
                return right_;
            } else {
                right_ = newNode;
                // 追加終了
                return null;
            }
        }
    }
    
    /**
     *  targetNodeノードを検索します。
     *  当メソッドは再帰を利用しているためStackOverFlowエラーが発生する危険があり、
     *  使用しないで下さい。
     *  @deprecated
     */
    TreeNode find(TreeNode targetNode) {
        int iComparison;
        
        iComparison = compare(targetNode);
        
        // このノードと同じ
        if (iComparison == 0) {
            return this;
        // このノードより小さい
        } else if (iComparison > 0) {
            if (left_ != null) {
                return left_.find(targetNode);
            } else {
                return null;
            }
        // このノードより大きい
        } else {
            if (right_ != null) {
                return right_.find(targetNode);
            } else {
                return null;
            }
        }
    }
    
    /**
     *  nodeContainerの先頭要素のノードを検索します。<br>
     *  nodeContainerの先頭要素のノードのキーとインスタンスのキーをコンパレータにより比較し、
     *  <ol>
     *      <li>このインスタンスより小さいキーの場合<br>
     *          nodeContainerの先頭要素としてこのインスタンスの左ノードを設定(上書き)し、
     *          {@link TreeNode#SMALLER}を返します。
     *      </li>
     *      <li>このインスタンスより大きいキーの場合<br>
     *          nodeContainerの先頭要素としてこのインスタンスの右ノードを設定(上書き)し、
     *          {@link TreeNode#BIGGER}を返します。
     *      </li>
     *      <li>このインスタンスと同じキーの場合<br>
     *          nodeContainerの先頭要素としてこのインスタンスを設定(上書き)し、
     *          {@link TreeNode#BINGO}を返します。
     *      </li>
     *      <li>見つからなかった場合<br>
     *          {@link TreeNode#NOTFOUND}を返します。
     *      </li>
     *  </ol>
     *  当メソッドは、内部で再帰を使用していません。
     *  そのため、呼び出し側で結果を判定しながらループさせる必要があります。
     *  @param nodeContainer 検索対象のノードを先頭要素として格納したリスト
     *  @return SMALLER/BIGGER/BINGO/NOTFOUND のいずれか
     */
    int find(List<TreeNode> nodeContainer) {
        int iComparison;
        
        iComparison = compare((TreeNode)nodeContainer.get(0));
        //iComparison = comp_.compare(this, nodeContainer.get(0));
        
        // このノードと同じ
        if (iComparison == 0) {
            nodeContainer.set(0, this);
            return BINGO;
        // このノードより小さい
        } else if (iComparison > 0) {
            if (left_ != null) {
                nodeContainer.set(0, left_);
                return SMALLER;
            } else {
                return NOTFOUND;
            }
        // このノードより大きい
        } else {
            if (right_ != null) {
                nodeContainer.set(0, right_);
                return BIGGER;
            } else {
                return NOTFOUND;
            }
        }
    }
}
